/* Copyright (C) 2012-2018 RealVNC Ltd. All Rights Reserved.
 *
 * This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */

#ifndef UUID_508ae798_92b2_bfd8_0648_fa3e9da865d8
#define UUID_508ae798_92b2_bfd8_0648_fa3e9da865d8

#include <VNCDeviceProviderBase.h>
#include "UsbDeviceInfoProvider.h"
#include <set>

struct libusb_context;
struct libusb_device;
struct libusb_config_descriptor;
struct libusb_interface_descriptor;

namespace vncdeviceprovider_usb
{
  class UsbDeviceProvider : public VNCDeviceProviderBase,
                            public UsbDeviceInfoObserver
  {
  public:
    UsbDeviceProvider(VNCDeviceProviderInterface *pInterface, 
              size_t interfaceSize,
              VNCDeviceProviderCallbacks *pCallbacks,
              size_t callbacksSize);
    virtual ~UsbDeviceProvider();

  private: // Device Provider API
    virtual VNCDiscoverySDKError setProperty(const char* pProperty,
        const char* pValue);
    virtual VNCDiscoverySDKError getProperty(const char *pProperty,
        char **ppValue);
    virtual void handleEvent(VNCDiscovererEventHandle* pHandle);
    virtual void handleTimeout(VNCDiscovererEventHandle* pHandle);
    virtual size_t getDeviceProperties(const VNCDeviceInfo *pDevice,
          VNCDiscoverySDKKeyValPair **ppProperties);
    virtual VNCDiscoverySDKError startMonitoring();
    virtual void startReporting();
    virtual VNCDiscoverySDKError stopMonitoring();
    virtual void stopReporting();

  private: // from UsbDeviceInfoObserver
    virtual void usbDeviceAdded(const UsbDevice& dev);
    virtual void usbDeviceRemoved(vnc_uint32_t id);
    virtual void setTimerForUsbDeviceInfoProvider(
              VNCDiscoverySDKTimeoutMicroseconds usecs);

  private:
    /// Populates an SDK device structure with the device information retrieved
    /// from libusb.
    ///
    /// This populates all the information about configurations/interfaces/end
    /// points.
    ///
    /// \param pLibusbCtx The libusb context.
    /// \param pDev The libusb device.
    /// \param pSysPath The system path.
    /// \param pSerial The serial number of the device (if known). This could
    /// be retrieved from libusb, but needs to be retrieved as a string
    /// descriptor. So if already available, it's better to make use of it
    /// (instead of re-retrieving).
    ///
    /// \return The new device, or NULL if the information can't be retrieved.
    VNCDeviceInfo* createDevice(libusb_context* pLibusbCtx,
        libusb_device* pDev, const char* pSysPath, const char* pSerial);

    /// Populates an SDK interface structure with the USB configuration
    /// information retrieved from libusb.
    ///
    /// This populates all the information about interfaces/end
    /// points.
    ///
    /// \param pLibusbCtx The libusb context.
    /// \param pConfig The libusb configuration.
    /// \param pSysPath The sys path of the device.
    /// \param isActive True if this is the currently active configuration
    /// \param configDesc Raw config descriptor from device, can be NULL
    ///
    /// \return The new configuration, or NULL if the information can't be
    /// retrieved.
    VNCDeviceInterface* createConfiguration(libusb_context* pLibusbCtx,
        libusb_config_descriptor* pConfig, const char* pSysPath, bool isActive,
        const unsigned char* configDesc, size_t configDescLength);

    /// Populates an SDK interface structure with the USB interface
    /// information retrieved from libusb.
    ///
    /// \param pLibusbCtx The libusb context.
    /// \param pIfrDescr The libusb interface descriptor.
    /// \param pSysPath The sys path of the device.
    /// \param isActive Whether the interface is part of the active
    /// configuration or not.
    ///
    /// \return The new interface, or NULL if the information can't be
    /// retrieved.
    VNCDeviceInterface* createInterface(libusb_context* pLibusbCtx,
        const libusb_interface_descriptor* pIfrDescr,
        const char* pSysPath, bool isActive, vnc_uint8_t configValue);

    /// Populates an SDK interface structure with the USB interface
    /// association information parsed from the descriptor
    ///
    /// \param start The point to start searching for the interface
    /// association descriptor. This is updated to the first
    /// non-examined byte upon return.
    /// \param end The end of valid descriptor data.
    /// \param isActive Whether the interface association is part of
    /// the active configuration or not.
    ///
    /// \return The new interface association, or NULL if the
    /// information can't be retrieved.
    VNCDeviceInterface* createInterfaceAssociation(
        const unsigned char*& start, const unsigned char* end,
        bool isActive);

    void findDevices();
    void checkExistingDevices();
    bool handleInEventHandles(VNCDiscovererEventHandle handle);
    VNCDeviceInfo* createDeviceDeepCopy(VNCDeviceInfo* device);
    /// \brief A new device was found while monitoring. Don't report it, but
    /// store it until the reporting is started again.
    ///
    /// \param device The device that was added. The function takes owneship of
    /// it.
    void deviceAddedWhileMonitoring(VNCDeviceInfo* device);
    /// \brief A new device was found while monitoring. Don't report it, but
    /// store it until the reporting is started again.
    ///
    /// If the removed device has already been reported, as soon as reporting
    /// is started again, report it's removal.
    ///
    /// If the removed device was added while monitoring, just remove it from
    /// the changed devices.
    void deviceRemovedWhileMonitoring(vnc_uint32_t id);
    /// \brief Reports the devices that have been added, or removed while
    /// monitoring (but not reporting).
    void reportChangedDevices();
    /// \brief Checks if a device is already known.
    ///
    /// The device may be in the known devices list, or in the changed devices
    /// list.
    bool isAddedDeviceDuplicate(vnc_uint32_t id) const;

  private:
    UsbDeviceInfoProvider mUsbDeviceInfoProvider;

    // This structure holds a VNCDeviceInfo, and indicates whether the device
    // is owned by the device provider.
    //
    // If owned == false, then device is a copy of a VNCDeviceInfo owned by
    // the Discovery SDK. It should be deleted normally.
    //
    // If owned == true, then device is owned by the device provider, and
    // must be deleted by calling freeDevice().
    struct KnownDevice
    {
      bool operator<(const KnownDevice &rhs) const
      { return device < rhs.device; }

      VNCDeviceInfo* device;
      bool owned;
    };

    typedef std::set<KnownDevice> KnownDevices;
    KnownDevices mKnownDevices;

    // Stores the devices changes while the device provider is monitoring. Once
    // reporting is enabled, the device changes will be reported back to the
    // SDK.
    typedef std::set<vnc_uint32_t> RemovedDevices;
    RemovedDevices mRemovedDevices;
    typedef std::set<VNCDeviceInfo*> AddedDevices;
    AddedDevices mAddedDevices;
  };
}

#endif /* !defined (UUID_508ae798_92b2_bfd8_0648_fa3e9da865d8) */

